/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   OneShot Resource Framework API Header File
::   Copyright   :   (C)2002-2009 Woodward
::   Platform(s) :   MPC5xx
::   Limitations :   None
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file Resource_OneShot.h
    \brief A One Shot output operates like a monostable. It can be setup to turn ON for a defined amount of
           time before turning OFF automatically. See \ref oneshotcontrol and \ref oneshotdiagnostics

           Useful sections
           - \ref oneshottrigger
           - \ref oneshotduration
           - \ref oneshotreport
           - \ref oneshotdiagnosticprinciples
           - \ref howtooneshotdiagnostics
*/

#ifndef __RESOURCE_ONESHOT_H
#define __RESOURCE_ONESHOT_H

/*----- INCLUDES ------------------------------------------------------------------------------------------*/
#include <typedefn.h>
#include <resource.h>

/*----- DEFINES -------------------------------------------------------------------------------------------*/
#define USE_ONESHOT_DIAG_REPORT_COND USE_ONESHOT_DIAG_REPORT

/*----- TYPEDEFS ------------------------------------------------------------------------------------------*/
#pragma pack(1)

/*! \brief Enumeration describes the set of mask definitions that are used to identify the valid one shot
           attributes in the S_OneShotResourceAttributes and S_OneShotCreateResourceAttributes data structures.

Each mask definition applies to a particular member of one of the data structures.
\code
    ...
    S_OneShotResourceAttributes AttribsObj;

    // Going to provide values for the resource condition and report condition so include in the mask
    AttribsObj.uValidAttributesMask = USE_ONESHOT_COND | USE_ONESHOT_REPORT_COND;
    AttribsObj.eResourceCondition = RES_ENABLED;
    AttribsObj.eReportCondition = RES_ENABLED;
    // The remainder of the data structure can be left undefined since the uValidAttributesMask
    //   indicates that the members are not valid
    ... \endcode */
typedef enum
{
    USE_ONESHOT_DEFAULTS = 0x0000,
    USE_ONESHOT_COND = 0x0001,                /*!< Select S_OneShotResourceAttributes::eResourceCondition */
    USE_ONESHOT_REPORT_COND = 0x0002,         /*!< Select S_OneShotResourceAttributes::eReportCondition */
    USE_ONESHOT_STATE = 0x0004,               /*!< Select S_OneShotResourceAttributes::eState */
    USE_ONESHOT_DURATION = 0x0008,            /*!< Select S_OneShotResourceAttributes::s4Duration */
    USE_ONESHOT_DIAG_REPORT = 0x0010,         /*!< Select S_OneShotCreateResourceAttributes::pfDiagCback */
    USE_ONESHOT_REPORT = 0x0020,              /*!< Select S_OneShotCreateResourceAttributes::pfReportCback */
    USE_ONESHOT_DYNAMIC_ON_CREATE = 0x0040,   /*!< Select S_OneShotCreateResourceAttributes::DynamicObj */
    USE_ONESHOT_DRIVER_MODE = 0x0080,         /*!< Select S_OneShotCreateResourceAttributes::eDriverMode */
    USE_ONESHOT_PWM_PORTION_1 = 0x0100,       /*!< Select S_OneShotResourceAttributes::PWM1Obj */
    USE_ONESHOT_PWM_PORTION_2 = 0x0200,       /*!< Select S_OneShotResourceAttributes::PWM2Obj */
    USE_ONESHOT_DURATION_2 = 0x0400,          /*!< Select S_OneShotResourceAttributes::s4Duration_2 */
    USE_ONESHOT_MAX_DURATION = 0x0800,        /*!< Select S_OneShotCreateResourceAttributes::uMaxExpectedDuration */
    USE_ONESHOT_REPORT_2 = 0x1000,            /*!< Select S_OneShotCreateResourceAttributes::pfAlternateReportCback */


/* IF THIS TYPE EXCEEDS 0x8000 THEN ALTER THE TYPE of OneShotAttributeMask_U ACCORDINGLY */

} E_OneShotAttributeMask;

typedef uint2 OneShotAttributeMask_U;

/*! \brief Diagnostic report information that is supplied as a parameter of the diagnostic report notification event */
typedef struct
{
    E_ModuleResource eOwningResource;   /*!< The framework resource that issued the report */
    E_ResourceDiagStatus eDiagStatus;   /*!< The diagnostic status of the connected actuator at this instant */
} S_OneShotDiagInfo;

/*! \brief Read only pointer type for the \ref S_OneShotDiagInfo information data structure that is issued with
           each oneshot diagnostic notification event */
typedef S_OneShotDiagInfo const* S_OneShotDiagInfoPtr;

/*! \brief Function pointer type that defines the prototype of a one shot diagnostic notification handler function.

The \c in_pDiagInfoObj parameter is a read only pointer to the diagnostic report associated with this notitification
while the \c in_uAppDataSentOnNotify parameter represents application data that was supplied when the report was defined.
It is sufficiently sized to allow it to hold a pointer variable. See S_OneShotCreateResourceAttributes::uDiagCbackUserData
and CreateResource_BEHAVIOUR_ONESHOT(). */
typedef void (*OneShotDiagCbackFuncPtrType)(S_OneShotDiagInfoPtr in_pDiagInfoObj, NativePtrSizedInt_U in_uAppDataSentOnNotify);

/*! \brief Function pointer type that defines the prototype of a one shot pulse complete notification handler function.

The \c in_eIssuingResource parameter describes the resource that issued this pulse just completed event 
while the \c in_uAppDataSentOnNotify parameter represents application data that was supplied when the report was defined.
It is sufficiently sized to allow it to hold a pointer variable. See S_OneShotCreateResourceAttributes::uReportCbackUserData
and CreateResource_BEHAVIOUR_ONESHOT(). */
typedef void (*OneShotReportCbackFuncPtrType)(E_ModuleResource in_eIssuingResource,NativePtrSizedInt_U in_uAppDataSentOnNotify);

/*! \brief Function pointer type that defines an alternative prototype of a one shot pulse complete notification handler function.

The \c in_eIssuingResource parameter describes the resource that issued this pulse just completed event 
while the \c in_uAppDataSentOnNotify parameter represents application data that was supplied when the report was defined.
It is sufficiently sized to allow it to hold a pointer variable. See S_OneShotCreateResourceAttributes::uReportCbackUserData
and CreateResource_BEHAVIOUR_ONESHOT(). The \p uPortionCompleted parameter identifies whether portion 0 or protion 1 just completed */
typedef void(*PFNONESHOTREPORTALTERNATE)(E_ModuleResource in_eIssuingResource, NativeVar_U uPortionCompleted, NativePtrSizedInt_U in_uAppDataSentOnNotify);

/*! \brief Describes the pulse width modulation attributes of a portion of the one shot */
typedef struct
{
     uint4 uFreqInHz;
     sint2 sDutyCycle;
} S_OneShotPWM;

/*! \brief This data structure describes all of the OneShot resource's runtime configuration attributes.

    The attributes are altered through the use of SetResourceAttributesBEHAVIOUR_ONESHOT(). The data structure
    does not need to be completely filled inorder to be used. The \b uValidAttributesMask is a bit field
    member that is used to identify which attributes are valid. Each attribute is identified with a separate
    bit mask that is logic-ORed into the a mask when an attribute is to be used.
\code
    ...
    S_OneShotResourceAttributes AttribsObj;

    // Going to provide values for the resource condition and report condition so include in the mask
    AttribsObj.uValidAttributesMask = USE_ONESHOT_COND | USE_ONESHOT_REPORT_COND;
    AttribsObj.eResourceCondition = RES_ENABLED;
    AttribsObj.eReportCondition = RES_ENABLED;
    // The remainder of the data structure can be left undefined since the uValidAttributesMask
    //   indicates that the members are not valid
    ... \endcode */
typedef struct
{
/*! Logic-OR the attributes [\ref USE_ONESHOT_COND, \ref USE_ONESHOT_REPORT_COND, \ref USE_ONESHOT_STATE,
    \ref USE_ONESHOT_DURATION] that are valid for this instance of the data structure */
    OneShotAttributeMask_U uValidAttributesMask;
/*! The condition of the resource. Trigger requests (see \ref oneshottrigger) will be ignored until the
    resource is set to \ref RES_ENABLED via this attribute. Diagnostics are also blocked while the resource is
    disabled. Select with the \ref USE_ONESHOT_COND bit mask. The default value of this attribute upon resource
    creation is \ref RES_DISABLED. */
    E_ResourceCond eResourceCondition;
/*! Allows the pulse complete notification function that was defined during resource creation to be issued
    [\ref RES_ENABLED] or blocked [\ref RES_DISABLED]. Select with the \ref USE_ONESHOT_REPORT_COND bit mask.
    This attribute is ignored if a notification report handler was not supplied during creation */
    E_ResourceCond eReportCondition;
    E_ResourceCond eResourceDiagCondition;
/*! Use this member to trigger the one shot to turn ON or to force a oneshot that is currently ON to terminate.
    See \ref oneshottrigger. Select with the \ref USE_ONESHOT_STATE bit mask. */
    E_ResourceState eState;
/*! Defines the length of time in microseconds that portion one of a trigger pulse is to remain ON for before
    switching to portion 2 (or turning OFF if portion 2 has zero duration). See \ref oneshotduration. Select
    with the \ref USE_ONESHOT_DURATION bit mask. */
    sint4 s4Duration;
/*! Defines the length of time in microseconds of portion 2 of the trigger pulse is to remain ON for before
    switching OFF. Portion 2 only occurs after portion 1 has expired. Select with the \ref USE_ONESHOT_DURATION_2
    bit mask. */
    sint4 s4Duration_2;
/*! When supported by the hardware, this object describes the pulse width modulation characteristics that portion
    1 of the trigger pulse shall take on while active. Select with the \ref USE_ONESHOT_PWM_PORTION_1 bit mask.
    This attribute defaults to give 100% (== ON) when not otherwise defined */
    S_OneShotPWM PWM1Obj;
/*! When supported by the hardware, this object describes the pulse width modulation characteristics that portion
    1 of the trigger pulse shall take on while active. Select with the \ref USE_ONESHOT_PWM_PORTION_1 bit mask.
    This attribute defaults to give 100% (== ON) when not otherwise defined */
    S_OneShotPWM PWM2Obj;
} S_OneShotResourceAttributes;

typedef S_OneShotResourceAttributes const* S_OneShotResourceAttributesPtr;

/*! \brief This data structure describes the creation attributes for a One Shot resource

    Included within the structure is the \p DynamicObj which represents the initial states for the resource's
    dynamic attributes. This data structure is referenced when the resource is created via the
    CreateResourceBEHAVIOUR_ONESHOT() function */
typedef struct
{
/*! Logic-OR the attributes [\ref USE_ONESHOT_DIAG_REPORT, \ref USE_ONESHOT_REPORT, \ref USE_ONESHOT_DYNAMIC_ON_CREATE]
    that are valid for this instance of the data structure. The framework will employ the creation defaults for those
    attributes that are left undefined. */
    OneShotAttributeMask_U uValidAttributesMask;
/*! Defines the dignostic notification callback function that the framework is to execute when new diagnostic
    information on the connected actuator is available. Selected with the \ref USE_ONESHOT_DIAG_REPORT, this
    attribute defaults to \c NULL when it is not selected. Thus the framework will not issue diagnostic reports
    if a diagnostic handler function is not provided. */
    OneShotDiagCbackFuncPtrType pfDiagCback;
/*! Defines the pulse complete notification callback function that the framework is to execute when a pulse
    completes and has been enabled. See \ref oneshotreport. Selected with the \ref USE_ONESHOT_REPORT, this
    attribute defaults to \c NULL when it is not selected. Thus the framework will not issue pulse complete
    reports, even if enabled, if a report handler function is not provided during creation. */
    OneShotReportCbackFuncPtrType pfReportCback;
/*! Application data that is supplied as a parameter of the S_OneShotCreateResourceAttributes::pfReportCback
    notification function. The type is sufficiently large enough to hold a pointer. */
    NativePtrSizedInt_U uReportCbackUserData;
/*! Application data that is supplied as a parameter of the S_OneShotCreateResourceAttributes::pfDiagCback
    notification function. The type is sufficiently large enough to hold a pointer. */
    NativePtrSizedInt_U uDiagCbackUserData;

/*! Initial values of the runtime attributes, which can be later altered through with SetResourceAttributesBEHAVIOUR_ONESHOT().
    Select with \ref USE_ONESHOT_DYNAMIC_ON_CREATE bit mask if some initialization values are supplied. */
    S_OneShotResourceAttributes DynamicObj;

/*! Some drivers have the facility to operate in more than one mode. This attribute allows the mode to be selected.
    Modes include a low side driver and push pull driver. Selected with \ref USE_ONESHOT_DRIVER_MODE, this attribute
    defaults to \ref DRIVER_LOW_SIDE in situations where mixed mode operation is supported or to the appropriate mode
    of the driver hardware when mixed mode operation is not supported. */
    E_DriverMode eDriverMode;

/*! Alternate report function that allows the portion of the pulse just completed to be communicated to
    the operator. A one shot with a portion 2 of zero shall not report the completion of this portion. Use the
    \ref USE_ONESHOT_REPORT_2 bit mask to employ this attribute. This attribute is not supported by default. */
    PFNONESHOTREPORTALTERNATE pfAlternateReportCback;

/*! Describes the expected max duration of a pulse in microseconds. This is used when the hardware is created to
    allow the most optimal settings to be chosen. Some hardware is able to support an output with much better
    granularity on the pulse when this value is sufficiently low enough. Select this value with the
    \ref USE_ONESHOT_MAX_DURATION bit mask. When not supplied the absolute maximum possible duration is assumed */
    uint4 uMaxExpectedDuration;

} S_OneShotCreateResourceAttributes;

typedef S_OneShotCreateResourceAttributes const* S_OneShotCreateResourceAttributesPtr;

#pragma pack()
/*----- EXTERNALS -----------------------------------------------------------------------------------------*/

/*----- PROTOTYPES ----------------------------------------------------------------------------------------*/
/* Optimised calls that can be applied to certain attribute sets                                           */
NativeError_S SetOneShotResourceDuration(E_ModuleResource in_eResource, sint4 in_s4Duration);
NativeError_S SetOneShotResourceState(E_ModuleResource in_eResource, E_ResourceState in_eState);

/* DO NOT use these functions directly. They are implicitly called through the use of the CreateResource() */
/*   and SetResourceAttributes() macro functions                                                           */
NativeError_S CreateResourceBEHAVIOUR_ONESHOT(E_ModuleResource in_eResource, S_OneShotCreateResourceAttributes const*);
NativeError_S DestroyResourceBEHAVIOUR_ONESHOT(E_ModuleResource in_eResource);
NativeError_S SetResourceAttributesBEHAVIOUR_ONESHOT(E_ModuleResource, S_OneShotResourceAttributes const*);

#endif /* __RESOURCE_ONESHOT_H */

/*----- END OF FILE ---------------------------------------------------------------------------------------*/
